from decimal import Decimal

from rest_framework import serializers, status
from rest_framework.response import Response
from rest_framework.validators import UniqueValidator
from back.models import User
from invoice.models import Payment
from partners.models import Partner
from settings.models import NumberingTemplate


class InvoiceSerializer(serializers.Serializer):
    id = serializers.ReadOnlyField(required=False, read_only=True)
    user = serializers.PrimaryKeyRelatedField(queryset=User.objects.all(), required=False)
    client = serializers.PrimaryKeyRelatedField(queryset=Partner.objects.all())
    invoice_number = serializers.CharField(max_length=128)
    items = serializers.JSONField(decoder=None, encoder=None, style={'base_template': 'textarea.html'})
    amount = serializers.DecimalField(decimal_places=2, max_digits=10, min_value=Decimal(0.01))
    date_issued = serializers.DateField()
    date_paid = serializers.DateField(required=False, allow_null=True)
    date_due = serializers.DateField(required=False, allow_null=True)
    status = serializers.ChoiceField(
        choices=[('pending', 'Pending'), ('paid', 'Paid'), ('overdue', 'Overdue'), ('cancelled', 'Cancelled')],
        required=False)
    description = serializers.CharField(allow_blank=True, required=False, style={'base_template': 'textarea.html'})
    type = serializers.CharField(allow_blank=True, required=False)
    image = serializers.ImageField(allow_null=True, required=False)
    client_name = serializers.CharField(source='client.name', read_only=True)
    image_deleted = serializers.BooleanField(default=False)

    template = serializers.PrimaryKeyRelatedField(
        queryset=NumberingTemplate.objects.all(),
        required=False,
        allow_null=True,
        help_text="ID of numbering template to use"
    )

    template_name = serializers.CharField(
        source='template.name',
        read_only=True,
        allow_null=True
    )

    def create(self, validated_data):
        return Payment.objects.create(**validated_data)

    image_deleted = serializers.BooleanField(
        required=False,
        default=False,
        write_only=True,
        help_text="Set to true to delete existing image"
    )

    def update(self, instance, validated_data):
        image_deleted = validated_data.pop('image_deleted', False)

        # Если установлен флаг удаления изображения
        if image_deleted and instance.image:
            instance.image.delete()  # Удаляем файл из хранилища
            instance.image = None  # Сбрасываем поле

        # Обновляем остальные поля
        for attr, value in validated_data.items():
            setattr(instance, attr, value)

        instance.save()
        return instance

    def validate(self, data):
        date_paid = data.get('date_paid')
        date_issued = data.get('date_issued')
        date_due = data.get('date_due')
        status_val = data.get('status')

        if self.instance:
            if date_issued is None:
                date_issued = self.instance.date_issued
            if status_val is None:
                status_val = self.instance.status

        if status_val == 'paid' and not date_paid:
            raise serializers.ValidationError({
                "status": "Status 'paid' requires date_paid field"
            })

        if date_paid and date_issued and date_paid < date_issued:
            raise serializers.ValidationError({
                "date_paid": "Payment date cannot be before issue date"
            })

        if date_issued and date_due and date_issued > date_due:
            raise serializers.ValidationError({
                "date_issued": "Issue date cannot be after due date"
            })

        return data
