import json
from datetime import datetime
from django.shortcuts import render, redirect
from rest_framework import generics, viewsets, status
from rest_framework.decorators import action, api_view
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView
from back.models import User
from settings.models import NumberingTemplate
from settings.serializers import ModelSerializerCompany_User, NumberingTemplateSerializer
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.contrib.auth import update_session_auth_hash
from django.http import JsonResponse


@api_view(http_method_names=['POST'])
def change_password(request):
    request_body = request.data
    data = request_body
    user = request.user

    current_password = data.get('current_password')
    new_password = data.get('new_password')
    confirm_password = data.get('confirm_password')

    if new_password != confirm_password:
        return Response(status=status.HTTP_400_BAD_REQUEST, data={
            'success': False,
            'error': 'Nauji slaptažodžiai nesutampa'
        })

    if not user.check_password(current_password):
        return Response(status=status.HTTP_400_BAD_REQUEST, data={
            'success': False,
            'error': 'Neteisingas dabartinis slaptažodis'
        })

    user.set_password(new_password)
    user.save()

    update_session_auth_hash(request, user)

    return Response(status=status.HTTP_200_OK, data={
        'success': True,
        'message': 'Slaptažodis sėkmingai pakeistas!'
    })


@login_required
def templates_for_user(request):
    return Response(status=status.HTTP_200_OK, data={NumberingTemplate.obects.filter(user=request.user)})


@login_required
def invoice_settings_view(request):
    templates = NumberingTemplate.objects.filter(user=request.user)

    if request.method == 'POST':
        form_type = request.POST.get('form_type')

        if form_type == 'template':
            try:
                template_id = request.POST.get('template_id')

                if template_id:
                    template = get_object_or_404(
                        NumberingTemplate,
                        id=template_id,
                        user=request.user
                    )
                else:
                    template = NumberingTemplate(user=request.user)

                template.name = request.POST.get('name')
                template.prefix = request.POST.get('prefix')
                template.next_number = int(request.POST.get('next_number', 1))
                template.format = request.POST.get('format')
                
                if template.format == 'custom':
                    custom_format = request.POST.get('custom_format', '').strip()
                    if not custom_format:
                        raise ValueError("Pritaikytas formatas yra privalomas!")
                    template.custom_format = custom_format
                else:
                    template.custom_format = ''

                template.save()
                messages.success(request, 'Šablonas sėkmingai išsaugotas!')
            except Exception as e:
                messages.error(request, f'Klaida išsaugant šabloną: {str(e)}')

        elif form_type == 'delete_template':
            try:
                template_id = request.POST.get('template_id')
                template = get_object_or_404(
                    NumberingTemplate,
                    id=template_id,
                    user=request.user
                )
                template.delete()
                messages.success(request, 'Šablonas sėkmingai ištrintas!')
            except Exception as e:
                messages.error(request, f'Klaida trinant šabloną: {str(e)}')
                
        return redirect('settings')

    now = datetime.now()
    return render(request, 'settings.html', {
        'numbering_templates': templates,
        'current_year': now.year,
        'current_month': f"{now.month:02d}"
    })


class UserNumberingTemplatesView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        templates = NumberingTemplate.objects.filter(user=request.user)
        serializer = NumberingTemplateSerializer(templates, many=True)
        return Response(serializer.data)


@login_required
def profile_view(request):
    user = request.user
    if request.method == 'POST':
        user.first_name = request.POST.get('first_name', '')
        user.last_name = request.POST.get('last_name', '')
        user.email = request.POST.get('email', '')
        user.phone = request.POST.get('phone', '')

        user.website = request.POST.get('website', '')
        user.email_company = request.POST.get('email_company', '')
        user.company = request.POST.get('company', '')
        user.company_type = request.POST.get('company_type', 'individual')
        user.position = request.POST.get('position', '')
        user.address = request.POST.get('address', '')
        user.bio = request.POST.get('bio', '')

        user.vat_code = request.POST.get('vat_code', '')
        user.registration_number = request.POST.get('registration_number', '')
        user.bank_name = request.POST.get('bank_name', '')
        user.bank_account = request.POST.get('bank_account', '')
        user.bic = request.POST.get('bic', '')

        if 'company_logo' in request.FILES:
            user.company_logo = request.FILES['company_logo']

        user.save()
        return redirect('profile')

    return render(request, 'profile.html', {'user': user})


class ModelViewCompany_User(generics.UpdateAPIView):
    permission_classes = [IsAuthenticated]
    queryset = User.objects.all()
    serializer_class = ModelSerializerCompany_User

    def get_object(self):
        return self.request.user


class NumberingTemplateViewSet(viewsets.ModelViewSet):
    serializer_class = NumberingTemplateSerializer
    permission_classes = [IsAuthenticated]

    @action(detail=False, methods=['get'])
    def list_for_user(self, request):
        templates = self.get_queryset()
        serializer = self.get_serializer(templates, many=True)
        return Response(serializer.data)

    def get_queryset(self):
        return NumberingTemplate.objects.filter(user=self.request.user)

    def perform_create(self, serializer):
        serializer.save(user=self.request.user)
