from rest_framework import serializers
from .models import Expense, Tag, ExpenseFile

class TagSerializer(serializers.ModelSerializer):
    class Meta:
        model = Tag
        fields = ['id', 'name', 'color']

class ExpenseFileSerializer(serializers.ModelSerializer):
    filename = serializers.SerializerMethodField()
    filesize = serializers.SerializerMethodField()

    class Meta:
        model = ExpenseFile
        fields = ['id', 'filename', 'filesize', 'uploaded_at']
        read_only_fields = ['uploaded_at']

    def get_filename(self, obj):
        return obj.filename

    def get_filesize(self, obj):
        return obj.filesize

class ExpenseSerializer(serializers.ModelSerializer):
    tags = TagSerializer(many=True, read_only=True)
    files = ExpenseFileSerializer(many=True, read_only=True)
    tag_ids = serializers.PrimaryKeyRelatedField(
        many=True,
        queryset=Tag.objects.all(),
        source='tags',
        write_only=True,
        required=False
    )
    uploaded_files = serializers.ListField(
        child=serializers.FileField(max_length=100000, allow_empty_file=False, use_url=False),
        write_only=True,
        required=False
    )

    class Meta:
        model = Expense
        fields = [
            'id', 'date', 'amount', 'description',
            'tags', 'files', 'tag_ids', 'uploaded_files',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['created_at', 'updated_at']

    def create(self, validated_data):
        uploaded_files = validated_data.pop('uploaded_files', [])
        tag_ids = validated_data.pop('tags', [])

        expense = Expense.objects.create(**validated_data)
        expense.tags.set(tag_ids)

        for file in uploaded_files:
            ExpenseFile.objects.create(expense=expense, file=file)

        return expense

    def update(self, instance, validated_data):
        uploaded_files = validated_data.pop('uploaded_files', [])
        tag_ids = validated_data.pop('tags', [])

        # Обновление основных полей
        instance.date = validated_data.get('date', instance.date)
        instance.amount = validated_data.get('amount', instance.amount)
        instance.description = validated_data.get('description', instance.description)
        instance.save()

        # Обновление тегов
        if tag_ids is not None:
            instance.tags.set(tag_ids)

        # Добавление новых файлов
        for file in uploaded_files:
            ExpenseFile.objects.create(expense=instance, file=file)

        return instance

class TagCreateSerializer(serializers.ModelSerializer):
    class Meta:
        model = Tag
        fields = ['id', 'name', 'color']
        extra_kwargs = {
            'name': {'validators': []},
        }