from django.db import models
from django.contrib.auth import get_user_model
from django.core.validators import FileExtensionValidator

User = get_user_model()


class Tag(models.Model):
    name = models.CharField(max_length=100, default='', verbose_name="Pavadinimas")
    color = models.CharField(max_length=7, default="#0d6efd", verbose_name="Spalva")
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="tags")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Žyma"
        verbose_name_plural = "Žymos"
        ordering = ['name']
        constraints = [
            models.UniqueConstraint(
                fields=['user', 'name'],
                name='unique_tag_per_user'
            )
        ]

    def __str__(self):
        return self.name


class Expense(models.Model):
    tags = models.ManyToManyField(Tag, blank=True, null=True, related_name="expenses", verbose_name="Žymos")
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="expenses")
    date = models.DateField(verbose_name="Data")
    amount = models.DecimalField(max_digits=10, decimal_places=2, verbose_name="Suma")
    description = models.TextField(blank=True, null=True, verbose_name="Aprašymas")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Išlaida"
        verbose_name_plural = "Išlaidos"
        ordering = ['-date']

    def __str__(self):
        return f"{self.date} - {self.amount}€"


class ExpenseFile(models.Model):
    expense = models.ForeignKey(Expense, on_delete=models.CASCADE, related_name="files")
    file = models.FileField(
        upload_to="expense_files/%Y/%m/%d/",
        verbose_name="Failas",
        validators=[
            FileExtensionValidator(allowed_extensions=['pdf', 'jpg', 'jpeg', 'png']),
        ]
    )
    uploaded_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Išlaidų failas"
        verbose_name_plural = "Išlaidų failai"

    def __str__(self):
        return self.file.name.split("/")[-1]

    @property
    def filename(self):
        return self.file.name.split("/")[-1]

    @property
    def filesize(self):
        size_bytes = self.file.size
        if size_bytes < 1024:
            return f"{size_bytes} B"
        elif size_bytes < 1024 * 1024:
            return f"{size_bytes / 1024:.1f} KB"
        else:
            return f"{size_bytes / (1024 * 1024):.1f} MB"
