from django.core.paginator import Paginator
from django.views.generic import TemplateView
from rest_framework import viewsets, status
from rest_framework.response import Response
from rest_framework.decorators import action
from rest_framework.pagination import PageNumberPagination
from django_filters.rest_framework import DjangoFilterBackend
from .models import Expense, Tag, ExpenseFile
from .serializers import ExpenseSerializer, TagSerializer, TagCreateSerializer
from django.db.models import Q
from django.http import FileResponse
import os
from rest_framework.permissions import IsAuthenticated  # Импорт класса аутентификации
from django.contrib.auth.mixins import LoginRequiredMixin

class StandardResultsSetPagination(PageNumberPagination):
    page_size = 10
    page_size_query_param = 'page_size'
    max_page_size = 100


class ExpenseViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]  # Требование авторизации
    serializer_class = ExpenseSerializer
    pagination_class = StandardResultsSetPagination
    filter_backends = [DjangoFilterBackend]
    filterset_fields = ['date']

    def get_queryset(self):
        user = self.request.user
        # Добавляем проверку is_authenticated для безопасности
        if not user.is_authenticated:
            return Expense.objects.none()
            
        queryset = Expense.objects.filter(user=user).order_by('-date')

        # Фильтрация по тексту
        search = self.request.query_params.get('search', None)
        if search:
            queryset = queryset.filter(
                Q(description__icontains=search)
            )

        # Фильтрация по тегам
        tags = self.request.query_params.get('tags', None)
        if tags:
            tag_ids = [int(tag_id) for tag_id in tags.split(',')]
            queryset = queryset.filter(tags__id__in=tag_ids).distinct()

        return queryset

    def perform_create(self, serializer):
        serializer.save(user=self.request.user)

    def create(self, request, *args, **kwargs):
        # Добавляем проверку аутентификации
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
                
        # Обработка тегов
        tag_ids = request.data.getlist('tag_ids', [])
        request.data._mutable = True
        request.data.setlist('tag_ids', tag_ids)
        request.data._mutable = False

        return super().create(request, *args, **kwargs)

    def update(self, request, *args, **kwargs):
        # Добавляем проверку аутентификации
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
                
        # Обработка тегов
        tag_ids = request.data.getlist('tag_ids', [])
        request.data._mutable = True
        request.data.setlist('tag_ids', tag_ids)
        request.data._mutable = False

        return super().update(request, *args, **kwargs)

    @action(detail=True, methods=['delete'], url_path='files/(?P<file_id>[^/.]+)')
    def delete_file(self, request, pk=None, file_id=None):
        # Проверка аутентификации
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
                
        try:
            expense = self.get_object()
            file = expense.files.get(id=file_id)
            file_path = file.file.path

            # Удаляем файл физически
            if os.path.exists(file_path):
                os.remove(file_path)

            file.delete()
            return Response(status=status.HTTP_204_NO_CONTENT)
        except ExpenseFile.DoesNotExist:
            return Response(
                {"detail": "Failas nerastas"},
                status=status.HTTP_404_NOT_FOUND
            )

    @action(detail=True, methods=['get'], url_path='files/(?P<file_id>[^/.]+)/download')
    def download_file(self, request, pk=None, file_id=None):
        # Проверка аутентификации
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
                
        try:
            expense = self.get_object()
            file = expense.files.get(id=file_id)
            return FileResponse(
                file.file.open(),
                as_attachment=True,
                filename=file.filename
            )
        except ExpenseFile.DoesNotExist:
            return Response(
                {"detail": "Failas nerastas"},
                status=status.HTTP_404_NOT_FOUND
            )


class TagViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]  # Требование авторизации
    serializer_class = TagSerializer
    pagination_class = None

    def get_queryset(self):
        user = self.request.user
        # Добавляем проверку is_authenticated
        if not user.is_authenticated:
            return Tag.objects.none()
            
        queryset = Tag.objects.filter(user=user).order_by('name')

        search = self.request.query_params.get('search')
        if search:
            queryset = queryset.filter(name__icontains=search)

        return queryset

    def get_serializer_class(self):
        if self.action in ['create', 'update', 'partial_update']:
            return TagCreateSerializer
        return TagSerializer

    def perform_create(self, serializer):
        serializer.save(user=self.request.user)


class ExpensesView(LoginRequiredMixin, TemplateView):
    template_name = 'spendings.html'
    login_url = '/login/'
    redirect_field_name = 'next'

    def get_context_data(self, **kwargs):
        # Явная проверка аутентификации
        if not self.request.user.is_authenticated:
            from django.core.exceptions import PermissionDenied
            raise PermissionDenied("Доступ запрещён")
            
        context = super().get_context_data(**kwargs)

        # Получение всех тегов текущего пользователя
        context['tags'] = Tag.objects.filter(user=self.request.user)

        # Фильтрация расходов
        expenses = Expense.objects.filter(user=self.request.user).order_by('-date')

        # Применение фильтров
        search = self.request.GET.get('search')
        if search:
            expenses = expenses.filter(description__icontains=search)

        date_filter = self.request.GET.get('date')
        if date_filter:
            expenses = expenses.filter(date=date_filter)

        tags_filter = self.request.GET.get('tags')
        if tags_filter:
            tag_ids = [int(tag_id) for tag_id in tags_filter.split(',')]
            expenses = expenses.filter(tags__id__in=tag_ids).distinct()

        # Пагинация
        paginator = Paginator(expenses, 10)
        page_number = self.request.GET.get('page')
        context['expenses'] = paginator.get_page(page_number)

        return context