import datetime
import re
from django.core.files.storage import default_storage
from django.shortcuts import redirect
from django.views.generic import TemplateView
from django.contrib.auth import login, logout
from django.contrib import messages
from partners.models import Partner
from settings.models import NumberingTemplate
from .forms import CustomUserCreationForm, CustomAuthenticationForm, User
import os
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from reportlab.lib.pagesizes import A4
from reportlab.lib import colors
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer, Image, PageBreak
from reportlab.lib.units import mm
from reportlab.lib.enums import TA_CENTER, TA_RIGHT, TA_LEFT
from io import BytesIO
from num2words import num2words
from django.contrib.auth.decorators import login_required
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont
from invoice.models import Payment
from django.utils import timezone
from datetime import timedelta, datetime
from django.db.models import Q, Sum, Count
import json

class LandingView(TemplateView):
    template_name = 'landing.html'


def register_view(request):
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        form = CustomUserCreationForm(request.POST)
        if form.is_valid():
            user = form.save(commit=False)
            user.username = user.email.split('@')[0]

            suffix = 1
            original_username = user.username
            while User.objects.filter(username=user.username).exists():
                user.username = f"{original_username}{suffix}"
                suffix += 1
            user.save()
            client = Partner.objects.create(
                user=user,
                name='Numatytasis klientas',
                contact_person='',
                email='',
                phone='',
                address='',
                vat_code='',
                registration_number='',
            )
            client.save()
            messages.success(request, 'Account created successfully!')
            return redirect('dashboard')
        else:
            messages.error(request, 'Please correct the errors below.')
    else:
        form = CustomUserCreationForm()

    return render(request, 'register.html', {'form': form})


def login_view(request):
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        form = CustomAuthenticationForm(request, data=request.POST)
        if form.is_valid():
            user = form.get_user()
            login(request, user)
            return redirect('dashboard')
        else:
            messages.error(request, 'Invalid email or password.')
    else:
        form = CustomAuthenticationForm()

    return render(request, 'login.html', {'form': form})


def logout_view(request):
    logout(request)
    messages.info(request, 'You have been logged out.')
    return redirect('landing')





@login_required
def dashboard_view(request):
    # Обработка фильтров
    selected_statuses = request.GET.getlist('status', [])
    date_start_str = request.GET.get('date_start')
    date_end_str = request.GET.get('date_end')

    # Период по умолчанию - последние 30 дней
    today = timezone.now().date()
    default_end = today
    default_start = today - timedelta(days=29)

    # Обработка дат
    date_start = default_start
    date_end = default_end

    if date_start_str:
        try:
            date_start = datetime.strptime(date_start_str, '%Y-%m-%d').date()
        except:
            pass

    if date_end_str:
        try:
            date_end = datetime.strptime(date_end_str, '%Y-%m-%d').date()
        except:
            pass

    # Базовый запрос
    payments = Payment.objects.filter(user=request.user)

    # Фильтр по статусам
    if selected_statuses:
        payments = payments.filter(status__in=selected_statuses)

    # Фильтр по дате выдачи
    payments = payments.filter(date_issued__gte=date_start, date_issued__lte=date_end)

    # График - динамически определяем группировку
    chart_data = []
    chart_labels = []

    days_diff = (date_end - date_start).days

    # Для периодов до 30 дней - группировка по дням
    if days_diff <= 30:
        for i in range(days_diff + 1):
            current_date = date_start + timedelta(days=i)
            day_payments = payments.filter(
                date_issued=current_date,
                status='paid'
            )
            total = day_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            chart_labels.append(current_date.strftime('%d %b'))

    # Для периодов 31-90 дней - группировка по неделям
    elif days_diff <= 90:
        current = date_start
        while current <= date_end:
            week_end = current + timedelta(days=6)
            if week_end > date_end:
                week_end = date_end

            week_payments = payments.filter(
                date_issued__gte=current,
                date_issued__lte=week_end,
                status='paid'
            )
            total = week_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            chart_labels.append(f"{current.strftime('%d %b')} - {week_end.strftime('%d %b')}")

            current = week_end + timedelta(days=1)

    # Для периодов более 90 дней - группировка по месяцам
    else:
        current = date_start.replace(day=1)
        while current <= date_end:
            next_month = current.replace(day=28) + timedelta(days=4)
            month_end = next_month - timedelta(days=next_month.day)
            if month_end > date_end:
                month_end = date_end

            month_payments = payments.filter(
                date_issued__gte=current,
                date_issued__lte=month_end,
                status='paid'
            )
            total = month_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            chart_labels.append(current.strftime('%b %Y'))

            current = month_end + timedelta(days=1)

    # Основные метрики
    metrics = payments.aggregate(
        total_revenue=Sum('amount', filter=Q(status='paid')),
        successful_transactions=Count('id', filter=Q(status='paid')),
        pending_invoices=Count('id', filter=Q(status='pending')),
        outstanding=Sum('amount', filter=Q(status='pending'))
    )

    total_revenue = metrics['total_revenue'] or 0
    successful_transactions = metrics['successful_transactions'] or 0
    pending_invoices = metrics['pending_invoices'] or 0
    outstanding = metrics['outstanding'] or 0

    # Рассчет процентного изменения выручки
    revenue_change_percent = 0
    if chart_data:
        if len(chart_data) > 1:
            last = chart_data[-1]
            prev = chart_data[-2]
            if prev > 0:
                revenue_change_percent = ((last - prev) / prev) * 100

    # Последние транзакции
    recent_transactions = payments.select_related('client').order_by('-date_issued')[:5]

    # Финансовый балл
    financial_score = 80
    if total_revenue > 10000:
        financial_score += 10
    elif total_revenue > 5000:
        financial_score += 5

    if successful_transactions > 20:
        financial_score += 10
    elif successful_transactions > 10:
        financial_score += 5

    if pending_invoices > 5:
        financial_score -= 10
    elif pending_invoices > 3:
        financial_score -= 5

    financial_score = max(0, min(100, financial_score))

    context = {
        'chart_labels': json.dumps(chart_labels),
        'chart_data': json.dumps(chart_data),
        'revenue_change_percent': revenue_change_percent,
        'total_revenue': total_revenue,
        'successful_transactions': successful_transactions,
        'pending_invoices': pending_invoices,
        'outstanding': outstanding,
        'recent_transactions': recent_transactions,
        'financial_score': financial_score,
        'selected_statuses': selected_statuses,
        'date_start': date_start.strftime('%Y-%m-%d'),
        'date_end': date_end.strftime('%Y-%m-%d'),
    }

    return render(request, 'dashboard.html', context)


@login_required
def profile_view(request):
    user = request.user
    if request.method == 'POST':
        user.first_name = request.POST.get('first_name', '')
        user.last_name = request.POST.get('last_name', '')
        user.email = request.POST.get('email', '')
        user.phone = request.POST.get('phone', '')
        user.website = request.POST.get('website', '')
        user.company = request.POST.get('company', '')
        user.position = request.POST.get('position', '')
        user.address = request.POST.get('address', '')
        user.bio = request.POST.get('bio', '')
        user.save()
        messages.success(request, 'Profile updated successfully!')
        return redirect('profile')

    return render(request, 'profile.html', {'user': user})


from django.core.paginator import Paginator
from django.shortcuts import render


def payment_list(request):
    payments = Payment.objects.filter(user=request.user).order_by('-date_issued')

    # Сохраняем параметры фильтрации
    filter_params = request.GET.copy()
    if 'page' in filter_params:
        del filter_params['page']

    # Применяем фильтры
    selected_statuses = request.GET.getlist('status', [])
    search_query = request.GET.get('search', '')
    client_id = request.GET.get('client', '')
    date_start = request.GET.get('date_start', '')
    date_end = request.GET.get('date_end', '')
    min_amount = request.GET.get('min_amount', '')
    max_amount = request.GET.get('max_amount', '')

    if selected_statuses:
        payments = payments.filter(status__in=selected_statuses)

    if search_query:
        payments = payments.filter(
            Q(client__name__icontains=search_query) |
            Q(invoice_number__icontains=search_query)
        )

    if client_id:
        payments = payments.filter(client_id=client_id)

    if date_start:
        payments = payments.filter(date_issued__gte=date_start)
    if date_end:
        payments = payments.filter(date_issued__lte=date_end)

    if min_amount:
        payments = payments.filter(amount__gte=min_amount)
    if max_amount:
        payments = payments.filter(amount__lte=max_amount)

    paginator = Paginator(payments, 10)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    clients = Partner.objects.filter(user=request.user)

    context = {
        'payments': page_obj,
        'clients': clients,
        'selected_statuses': selected_statuses,
        'filter_params': filter_params.urlencode(),  # Добавляем параметры фильтрации
        'request': request,
    }
    return render(request, 'payments.html', context)


@login_required
def payments(request):
    payments = Payment.objects.filter(user=request.user).order_by('-date_issued')
    clients = Partner.objects.filter(user=request.user)

    # Пагинация
    paginator = Paginator(payments, 10)
    page_number = request.GET.get('page')
    payments = paginator.get_page(page_number)  # Переименовали в payments
    print(payments)
    return render(request, 'payments.html', {
        'payments': payments,  # Теперь это объект страницы
        'clients': clients,
        'day_today': datetime.date.today(),
    })


@login_required
def register_support_fonts(request):
    try:
        # Попробуем найти шрифт в нескольких местах
        font_locations = [
            # Windows
            '/home/apskaita/public_html/finance/back/fonts/DejaVuSans.ttf',  # Arial Unicode MS
            'C:/Windows/Fonts/times.ttf',  # Arial Unicode MS',  # Times New Roman

            # Linux
            '/usr/share/fonts/truetype/dejavu/',
            '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',

            # MacOS
            '/Library/Fonts/Arial Unicode.ttf',

            # В проекте
            os.path.join(os.path.dirname(__file__), 'fonts\\arialuni.ttf'),
            os.path.join(os.path.dirname(__file__), 'fonts\\times.ttf'),
        ]
        print('new')
        for font_path in font_locations:
            print(2)
            if os.path.exists(font_path):
                print(font_path)
                font_name = os.path.splitext(os.path.basename(font_path))[0]
                try:
                    pdfmetrics.registerFont(TTFont(font_name, font_path))
                    # Регистрируем жирную версию если есть
                    bold_path = '/home/apskaita/public_html/finance/back/fonts/arial_bolditalicmt.ttf'
                    if os.path.exists(bold_path):
                        pdfmetrics.registerFont(TTFont(f'{font_name}-Bold', bold_path))
                    return font_name
                except:
                    continue

        # Если ничего не найдено, используем стандартный шрифт
        return 'Helvetica'
    except Exception as e:
        return 'Helvetica'


@login_required
def generate_invoice_pdf(request, payment_id):
    payment = get_object_or_404(Payment, id=payment_id, user=request.user)

    # Проверяем кастомный номер инвойса из параметра запроса
    custom_invoice_number = request.GET.get('invoice_number')
    if custom_invoice_number:
        payment.invoice_number = custom_invoice_number

    # Остальной код остается без изменений...
    client = payment.client

    # Lithuanian month names
    def get_lithuanian_month(month):
        months = {
            1: "sausio", 2: "vasario", 3: "kovo", 4: "balandžio",
            5: "gegužės", 6: "birželio", 7: "liepos", 8: "rugpjūčio",
            9: "rugsėjo", 10: "spalio", 11: "lapkričio", 12: "gruodžio"
        }
        return months.get(month, "")

    # Register fonts
    support_font = register_support_fonts(request)

    # Create PDF buffer
    buffer = BytesIO()

    # Document settings
    doc = SimpleDocTemplate(
        buffer,
        pagesize=A4,
        leftMargin=20 * mm,
        rightMargin=20 * mm,
        topMargin=10 * mm,
        bottomMargin=10 * mm,
        title=f"Sąskaita {payment.invoice_number}"
    )

    # Styles
    styles = getSampleStyleSheet()

    # Custom styles
    style_heading = ParagraphStyle(
        'Heading1',
        parent=styles['Heading1'],
        alignment=TA_LEFT,
        fontSize=14,
        spaceAfter=5,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold'
    )

    style_normal = ParagraphStyle(
        'Normal',
        parent=styles['BodyText'],
        fontSize=10,
        spaceAfter=3,
        fontName=support_font
    )

    style_bold = ParagraphStyle(
        'Bold',
        parent=style_normal,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold'
    )

    style_table_header = ParagraphStyle(
        'TableHeader',
        parent=style_normal,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold',
        fontSize=9,
        alignment=TA_CENTER
    )

    style_table_cell = ParagraphStyle(
        'TableCell',
        parent=style_normal,
        fontSize=9
    )

    style_table_cell_center = ParagraphStyle(
        'TableCellCenter',
        parent=style_table_cell,
        alignment=TA_CENTER
    )

    style_table_cell_right = ParagraphStyle(
        'TableCellRight',
        parent=style_table_cell,
        alignment=TA_RIGHT
    )

    # Currency formatting
    def format_currency(value):
        try:
            value = float(value)
            return f"{value:,.2f}".replace(",", " ").replace(".", ",")
        except:
            return "0,00"

    # Start building the document
    story = []

    # 1. Header with logo
    logo = None
    if request.user.company_logo:
        try:
            with default_storage.open(request.user.company_logo.name, 'rb') as logo_file:
                logo_data = BytesIO(logo_file.read())
                ext = request.user.company_logo.name.split('.')[-1].lower()
                if ext in ['jpg', 'jpeg', 'png']:
                    logo = Image(logo_data, width=100, height=30)
        except Exception as e:
            print(f"Logo error: {e}")

    header_table_data = []
    name = "SĄSKAITA FAKTŪRA"
    template = payment.template
    if payment.template.id in [1, 2, 3, 4]:
        name = NumberingTemplate.objects.all().get(pk=template.id).name

    if logo:
        header_table_data.append([Paragraph(name, style_heading), logo])
    else:
        header_table_data.append([Paragraph(name, style_heading), ""])

    header_table = Table(header_table_data, colWidths=[120 * mm, 50 * mm])
    header_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('ALIGN', (0, 0), (0, 0), 'LEFT'),
        ('ALIGN', (1, 0), (1, 0), 'RIGHT'),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 10),
    ]))
    story.append(header_table)
    story.append(Spacer(1, 5))

    # 2. Invoice number and date
    issued_month_lt = get_lithuanian_month(payment.date_issued.month)
    number_date_data = [
        [
            Paragraph(f"<b>Serija ir Nr.:</b> {payment.invoice_number}", style_normal),
            Paragraph(
                f"<b>Išrašymo data:</b> {payment.date_issued.year} m. {issued_month_lt} {payment.date_issued.day} d.",
                style_normal
            )
        ]
    ]
    number_date_table = Table(number_date_data, colWidths=[90 * mm, 80 * mm])
    number_date_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('ALIGN', (0, 0), (0, 0), 'LEFT'),
        ('ALIGN', (1, 0), (1, 0), 'RIGHT'),
    ]))
    story.append(number_date_table)
    story.append(Spacer(1, 15))

    # 3. Seller and buyer information in bordered boxes
    seller_info = [Paragraph("<b>Pardavėjas</b>", style_bold),
                   Paragraph(request.user.company or f"{request.user.first_name} {request.user.last_name}",
                             style_normal)]
    if request.user.address:
        seller_info.append(Paragraph(f"Adresas: {request.user.address}", style_normal))
    if request.user.registration_number:
        seller_info.append(Paragraph(f"Įmonės kodas: {request.user.registration_number}", style_normal))
    if request.user.vat_code:
        seller_info.append(Paragraph(f"PVM kodas: {request.user.vat_code}", style_normal))
    if request.user.website:
        seller_info.append(Paragraph(f"Svetainė: {request.user.website}", style_normal))
    if request.user.email_company:
        seller_info.append(Paragraph(f"El. paštas: {request.user.email_company}", style_normal))
    if request.user.address:
        seller_info.append(Paragraph(f"Telefonas: {request.user.address}", style_normal))
    if request.user.company:
        seller_info.append(Paragraph(f"Įmonės pavadinimas: {request.user.company}", style_normal))
    if request.user.bank_name:
        seller_info.append(Paragraph(f"Banko pavadinimas: {request.user.bank_name}", style_normal))
    if request.user.bank_account:
        seller_info.append(Paragraph(f"Banko sąskaita: {request.user.bank_account}", style_normal))
    if request.user.bic:
        seller_info.append(Paragraph(f"BIC/SWIFT kodas: {request.user.bic}", style_normal))

    buyer_info = [Paragraph("<b>Pirkėjas</b>", style_bold)]
    if client:
        buyer_info.append(Paragraph(client.name, style_normal))
        if client.address:
            buyer_info.append(Paragraph(f"Adresas: {client.address}", style_normal))
        if client.registration_number:
            buyer_info.append(Paragraph(f"Įmonės kodas: {client.registration_number}", style_normal))
        if client.vat_code:
            buyer_info.append(Paragraph(f"PVM kodas: {client.vat_code}", style_normal))
        if client.email:
            buyer_info.append(Paragraph(f"El. paštas: {client.email}", style_normal))
        if client.contact_person:
            buyer_info.append(Paragraph(f"Kontaktinis asmuo: {client.contact_person}", style_normal))
        if client.phone:
            buyer_info.append(Paragraph(f"Telefonas: {client.phone}", style_normal))
    else:
        buyer_info.append(Paragraph("Nenurodytas", style_normal))

    # Create bordered boxes using table
    seller_buyer_data = [[seller_info, buyer_info]]
    seller_buyer_table = Table(
        seller_buyer_data,
        colWidths=[85 * mm, 85 * mm]
    )

    # Box styling with borders
    seller_buyer_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('BOX', (0, 0), (0, 0), 1, colors.black),  # Left box border
        ('BOX', (1, 0), (1, 0), 1, colors.black),  # Right box border
        ('PADDING', (0, 0), (-1, -1), 5),
        ('LEFTPADDING', (0, 0), (-1, -1), 3),
        ('RIGHTPADDING', (0, 0), (-1, -1), 3),
        ('TOPPADDING', (0, 0), (-1, -1), 5),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 5),
    ]))

    story.append(seller_buyer_table)
    story.append(Spacer(1, 15))

    # 4. Items table
    table_data = [
        [
            Paragraph("Pavadinimas", style_table_header),
            Paragraph("Vnt. kaina (be PVM)", style_table_header),
            Paragraph("Kiekis", style_table_header),
            Paragraph("Matas", style_table_header),
            Paragraph("Suma be VPM", style_table_header),
        ]
    ]

    # Add discount column if needed
    no_discount = True
    if any([i.get('discount', 0) for i in payment.items]):
        no_discount = False
        table_data[0].insert(5, Paragraph("Nuolaida", style_table_header))

    no_vat = True
    if any([i.get('vat', 0) for i in payment.items]):
        no_vat = False
        table_data[0].insert(1, Paragraph("PVM", style_table_header))

    # Calculate column widths
    col_widths = [70 * mm, 15 * mm, 25 * mm, 15 * mm, 15 * mm, 25 * mm]
    if not no_discount:
        col_widths.insert(5, 20 * mm)  # Discount column

    # Add items
    vpm = 0
    pre_total_to_tittle = 0
    for item in payment.items:
        name = item.get('name', '')
        quantity = item.get('quantity', 1)
        unit = item.get('unit', '')
        vat = item.get('vat', '')
        disc = item.get('discount', 0)
        price = item.get('unit_price', 0)
        pre_total = price * quantity
        pre_total2 = pre_total - (pre_total / 100 * disc)
        pre_total_to_tittle += pre_total2
        vpm += (pre_total2 * (item.get('vat', 0) / 100))

        row = [
            Paragraph(name, style_table_cell_center),
            Paragraph(f"{format_currency(price)} €", style_table_cell_center),
            Paragraph(str(quantity), style_table_cell_center),
            Paragraph(unit, style_table_cell_center),
            Paragraph(f"{format_currency(pre_total2)} €", style_table_cell_center),
        ]
        if not no_vat:
            row.insert(1, Paragraph(f"{vat}%", style_table_cell_center))
        if not no_discount:
            row.insert(5, Paragraph(f"{disc}%", style_table_cell_center))
        table_data.append(row)

    # Create items table
    items_table = Table(table_data, colWidths=col_widths)
    items_table.setStyle(TableStyle([
        ('GRID', (0, 0), (-1, -1), 1, colors.black),
        ('FONTSIZE', (0, 0), (-1, -1), 9),
        ('BOTTOMPADDING', (0, 0), (-1, 0), 5),
        ('TOPPADDING', (0, 0), (-1, 0), 5),
        ('ALIGN', (2, 0), (-1, -1), 'RIGHT'),
        ('ALIGN', (1, 0), (1, -1), 'CENTER'),
        ('ALIGN', (3, 0), (4, -1), 'CENTER'),
    ]))
    story.append(items_table)
    story.append(Spacer(1, 10))

    # 5. Totals
    total_amount = float(payment.amount)
    if not no_vat:
        story.append(Paragraph(f"Iš viso (be PVM): {format_currency(pre_total_to_tittle)} €", style_table_cell_right))
        story.append(Paragraph(f"PVM: {format_currency(vpm)} €", style_table_cell_right))
        story.append(
            Paragraph(f"Iš viso (su PVM): {format_currency(pre_total_to_tittle + vpm)} €", style_table_cell_right))
    else:
        story.append(Paragraph(f"Iš viso: {format_currency(pre_total_to_tittle + vpm)} €", style_table_cell_right))
    story.append(Spacer(1, 10))

    # 6. Description and payment terms
    story.append(Paragraph(f"Aprašymas: {payment.description}", style_normal))

    # 7. Amount in words
    try:
        amount_in_words = num2words(
            total_amount,
            lang='lt',
            to='currency',
            currency='EUR'
        ).capitalize().replace('eurai', 'Eur').replace('euro', 'Eur')
        story.append(Paragraph(f"Suma žodžiais: {amount_in_words}", style_normal))
    except Exception as e:
        print(f"Num2words error: {e}")

    if payment.date_due:
        due_month_lt = get_lithuanian_month(payment.date_due.month)
        story.append(Paragraph(
            f"Apmokėti iki: {payment.date_due.year} m. {due_month_lt} {payment.date_due.day} d.",
            style_normal
        ))

    story.append(Spacer(1, 15))

    # 8. Service image on new page
    service_image = None
    if payment.image:
        try:
            with default_storage.open(payment.image.name, 'rb') as img_file:
                img_data = BytesIO(img_file.read())

                # Resize with PIL
                try:
                    from PIL import Image as PILImage
                    pil_img = PILImage.open(img_data)
                    img_width, img_height = pil_img.size

                    # Calculate proportional size
                    max_width = 180 * mm
                    max_height = 200 * mm
                    width_ratio = max_width / img_width
                    height_ratio = max_height / img_height
                    scale_ratio = min(width_ratio, height_ratio, 1.0)

                    new_width = img_width * scale_ratio
                    new_height = img_height * scale_ratio

                    img_data.seek(0)
                    service_image = Image(img_data, width=new_width, height=new_height)
                except:
                    img_data.seek(0)
                    service_image = Image(img_data, width=150 * mm, height=100 * mm)
        except Exception as e:
            print(f"Service image error: {e}")

    if service_image:
        story.append(PageBreak())  # Force new page
        story.append(Spacer(1, 5))
        service_image.hAlign = 'CENTER'
        story.append(service_image)

    # Generate PDF
    doc.build(story)

    # Prepare response
    company_name = request.user.company or request.user.get_full_name() or "Faktura"
    safe_company_name = re.sub(r'[^\w\sąčęėįšųūžĄČĘĖĮŠŲŪŽ]', '', company_name)
    safe_company_name = re.sub(r'\s+', '_', safe_company_name)[:50]
    filename = f"Faktūra_{safe_company_name}.pdf"

    buffer.seek(0)
    response = HttpResponse(buffer, content_type='application/pdf')

    if 'download' in request.GET:
        response['Content-Disposition'] = f'attachment; filename="{filename}"'
    else:
        response['Content-Disposition'] = f'inline; filename="{filename}"'

    return response


def profile(request):
    return render(request, template_name='profile.html')
