from datetime import datetime

from django.db.models import Q
from django.shortcuts import render, redirect
from rest_framework import generics, viewsets, status
from rest_framework.decorators import action
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from back.models import User
from settings.models import NumberingTemplate
from settings.serializers import ModelSerializerCompany_User, NumberingTemplateSerializer
from django.contrib import messages
from django.contrib.auth.decorators import login_required


def invoice_settings_view(request):
    # Получаем настройки и шаблоны
    templates = NumberingTemplate.objects.filter(user=request.user)

    if request.method == 'POST':
        form_type = request.POST.get('form_type')

        # Обработка общих настроек

        # Обработка шаблонов нумерации
        if form_type == 'template':
            try:
                template_id = request.POST.get('template_id')

                if template_id:
                    # Редактирование существующего шаблона
                    template = get_object_or_404(
                        NumberingTemplate,
                        id=template_id,
                        user=request.user
                    )
                else:
                    # Создание нового шаблона
                    template = NumberingTemplate(user=request.user)

                # Обновление данных шаблона
                template.name = request.POST.get('name')
                template.prefix = request.POST.get('prefix')
                template.next_number = int(request.POST.get('next_number', 1))
                template.format = request.POST.get('format')
                template.custom_format = request.POST.get('custom_format', '')

                template.save()
                messages.success(request, 'Template saved successfully!')
            except Exception as e:
                messages.error(request, f'Error saving template: {str(e)}')

        elif form_type == 'delete_template':
            try:
                template_id = request.POST.get('template_id')
                template = get_object_or_404(
                    NumberingTemplate,
                    id=template_id,
                    user=request.user
                )
                template.delete()
                messages.success(request, 'Template deleted successfully!')
            except Exception as e:
                messages.error(request, f'Error deleting template: {str(e)}')
        return redirect('settings')

    # Получаем текущий год и месяц для шаблона
    now = datetime.now()
    return render(request, 'settings.html', {
        'numbering_templates': templates,
        'current_year': now.year,
        'current_month': f"{now.month:02d}"
    })


@login_required
def profile_view(request):
    user = request.user
    if request.method == 'POST':
        # Обновляем все поля пользователя
        user.first_name = request.POST.get('first_name', '')
        user.last_name = request.POST.get('last_name', '')
        user.email = request.POST.get('email', '')
        user.phone = request.POST.get('phone', '')

        # Обязательно добавляем новые поля
        user.website = request.POST.get('website', '')
        user.email_company = request.POST.get('email_company', '')
        user.company = request.POST.get('company', '')
        user.company_type = request.POST.get('company_type', 'individual')
        user.position = request.POST.get('position', '')
        user.address = request.POST.get('address', '')
        user.bio = request.POST.get('bio', '')

        # Финансовые поля
        user.vat_code = request.POST.get('vat_code', '')
        user.registration_number = request.POST.get('registration_number', '')
        user.bank_name = request.POST.get('bank_name', '')
        user.bank_account = request.POST.get('bank_account', '')
        user.bic = request.POST.get('bic', '')

        # Обработка загрузки логотипа
        if 'company_logo' in request.FILES:
            user.company_logo = request.FILES['company_logo']

        user.save()
        return redirect('profile')

    return render(request, 'profile.html', {'user': user})


class ModelViewCompany_User(generics.UpdateAPIView):
    queryset = User.objects.all()
    serializer_class = ModelSerializerCompany_User

    def get_object(self):
        return self.request.user

class NumberingTemplateViewSet(viewsets.ModelViewSet):
    serializer_class = NumberingTemplateSerializer
    permission_classes = [IsAuthenticated]

    @action(detail=False, methods=['get'])
    def list_for_user(self, request):
        templates = self.get_queryset()
        serializer = self.get_serializer(templates, many=True)
        return Response(serializer.data)

    def get_queryset(self):
        return NumberingTemplate.objects.filter(
            Q(user=self.request.user) |
            Q(id__in=[1, 2, 3, 4])
        )

    def perform_create(self, serializer):
        serializer.save(user=self.request.user)
