import re
import os
from django.db.models import Min, Max
import datetime
import json
from spendings.models import Tag
from django.core.files.storage import default_storage
from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.http import require_POST
from django.views.generic import TemplateView
from django.contrib.auth import login, logout
from django.contrib import messages
from partners.models import Partner
from settings.models import NumberingTemplate
from spendings.models import Tag, Expense
from .forms import CustomUserCreationForm, CustomAuthenticationForm, User
from django.http import HttpResponse, JsonResponse
from reportlab.lib.pagesizes import A4
from reportlab.lib import colors
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer, Image, PageBreak
from reportlab.lib.units import mm
from reportlab.lib.enums import TA_CENTER, TA_RIGHT, TA_LEFT
from io import BytesIO
from num2words import num2words
from django.contrib.auth.decorators import login_required
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont
from invoice.models import Payment
from django.utils import timezone
from datetime import timedelta
from django.db.models import Q, Sum, Count

LITHUANIAN_MONTHS = {
    1: "sausio", 2: "vasario", 3: "kovo", 4: "balandžio",
    5: "gegužės", 6: "birželio", 7: "liepos", 8: "rugpjūčio",
    9: "rugsėjo", 10: "spalio", 11: "lapkričio", 12: "gruodžio"
}


def custom_page_not_found(request, exception):
    # Ваш кастомный рендеринг 404 страницы
    return render(request, '404.html', status=404)
def format_date_lt(date_obj):
    return f"{date_obj.day} {LITHUANIAN_MONTHS[date_obj.month]} {date_obj.year} m."


class LandingView(TemplateView):
    template_name = 'landing.html'


def register_view(request):
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        form = CustomUserCreationForm(request.POST)
        if form.is_valid():
            user = form.save(commit=False)
            user.username = user.email.split('@')[0]

            suffix = 1
            original_username = user.username
            while User.objects.filter(username=user.username).exists():
                user.username = f"{original_username}{suffix}"
                suffix += 1
            user.save()

            client = Partner.objects.create(
                user=user,
                name='Klientas',
                contact_person='',
                email='',
                phone='',
                address='',
                vat_code='',
                registration_number='',
            )
            template1 = NumberingTemplate.objects.create(
                user=user,
                name='Sąskaita Faktūra',
                prefix='SF',
                next_number=1,
                format='month',
            )
            template2 = NumberingTemplate.objects.create(
                user=user,
                name='Išankstinė sąskaita faktūra',
                prefix='SF',
                next_number=1,
                format='month',
            )
            template3 = NumberingTemplate.objects.create(
                user=user,
                name='PVM Sąskaita Faktūra',
                prefix='SF',
                next_number=1,
                format='month',
            )
            template4 = NumberingTemplate.objects.create(
                user=user,
                name='Komercinis pasiūlymas',
                prefix='SF',
                next_number=1,
                format='month',
            )
            template1.save()
            template2.save()
            template3.save()
            template4.save()
            client.save()
            for i in [
                ['Darbuotojai', user, '#1cc94a'],
                ['Biuras', user, '#cc1dc6'],
                ['Atlyginimai', user, '#6b25a1'],
                ['Freelanceriai', user, '#cc721d'],
                ['Komunaliniai mokesčiai', user, '#3dcc1d'],
                ['Kuras', user, '#cc1dac'],
                ['Facebook / Google reklamos', user, '#1d69cc'],
                ['Reklama', user, '#1da0cc'],
                ['Maistas', user, '#cc1d77'],
                ['Verslo išlaidos', user, '#1dcc75']
            ]:
                a = Tag.objects.create(
                    name=i[0],
                    user=i[1],
                    color=i[2],
                )
                a.save()
            messages.success(request, 'Account created successfully!')
            return redirect('dashboard')
        else:
            messages.error(request, 'Please correct the errors below.')
    else:
        form = CustomUserCreationForm()

    return render(request, 'register.html', {'form': form})


def login_view(request):
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        form = CustomAuthenticationForm(request, data=request.POST)
        if form.is_valid():
            user = form.get_user()
            login(request, user)
            return redirect('dashboard')
        else:
            messages.error(request, 'Invalid email or password.')
    else:
        form = CustomAuthenticationForm()

    return render(request, 'login.html', {'form': form})


def logout_view(request):
    logout(request)
    messages.info(request, 'You have been logged out.')
    return redirect('landing')

LITHUANIAN_MONTHS = {
    1: 'Sausio', 2: 'Vasario', 3: 'Kovo', 4: 'Balandžio',
    5: 'Gegužės', 6: 'Birželio', 7: 'Liepos', 8: 'Rugpjūčio',
    9: 'Rugsėjo', 10: 'Spalio', 11: 'Lapkričio', 12: 'Gruodžio'
}


def format_date_lt(date):
    return f"{date.day} {LITHUANIAN_MONTHS[date.month]} {date.year}"


@login_required
def dashboard_view(request):
    selected_statuses = request.GET.getlist('status', [])
    date_start_str = request.GET.get('date_start')
    date_end_str = request.GET.get('date_end')

    payments = Payment.objects.filter(user=request.user)

    if selected_statuses:
        payments = payments.filter(status__in=selected_statuses)

    date_start = None
    date_end = None
    all_time = False

    if date_start_str:
        try:
            date_start = datetime.datetime.strptime(date_start_str, '%Y-%m-%d').date()
        except:
            pass

    if date_end_str:
        try:
            date_end = datetime.datetime.strptime(date_end_str, '%Y-%m-%d').date()
        except:
            pass

    if not date_start_str and not date_end_str:
        all_time = True
        first_payment = payments.order_by('date_issued').first()
        last_payment = payments.order_by('-date_issued').first()

        if first_payment and last_payment:
            date_start = first_payment.date_issued
            date_end = last_payment.date_issued
        else:
            today = timezone.now().date()
            date_start = today - timedelta(days=29)
            date_end = today
    else:
        if date_start and date_end:
            payments = payments.filter(date_issued__range=[date_start, date_end])
        elif date_start:
            payments = payments.filter(date_issued__gte=date_start)
        elif date_end:
            payments = payments.filter(date_issued__lte=date_end)

    if not date_start or not date_end:
        first_payment = payments.order_by('date_issued').first()
        last_payment = payments.order_by('-date_issued').first()

        date_start = first_payment.date_issued if first_payment else (timezone.now().date() - timedelta(days=29))
        date_end = last_payment.date_issued if last_payment else timezone.now().date()

    days_diff = (date_end - date_start).days

    chart_data = []
    chart_labels = []

    if days_diff <= 30:
        for i in range(days_diff + 1):
            current_date = date_start + timedelta(days=i)
            day_payments = payments.filter(
                date_issued=current_date,
                status='paid'
            )
            total = day_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            chart_labels.append(f"{current_date.day} {LITHUANIAN_MONTHS[current_date.month]}")
    elif days_diff <= 90:
        current = date_start
        while current <= date_end:
            week_end = current + timedelta(days=6)
            if week_end > date_end:
                week_end = date_end

            week_payments = payments.filter(
                date_issued__gte=current,
                date_issued__lte=week_end,
                status='paid'
            )
            total = week_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            start_str = f"{current.day} {LITHUANIAN_MONTHS[current.month]}"
            end_str = f"{week_end.day} {LITHUANIAN_MONTHS[week_end.month]}"
            chart_labels.append(f"{start_str} - {end_str}")

            current = week_end + timedelta(days=1)
    else:
        current = date_start.replace(day=1)
        while current <= date_end:
            next_month = current.replace(day=28) + timedelta(days=4)
            month_end = next_month - timedelta(days=next_month.day)
            if month_end > date_end:
                month_end = date_end

            month_payments = payments.filter(
                date_issued__gte=current,
                date_issued__lte=month_end,
                status='paid'
            )
            total = month_payments.aggregate(total=Sum('amount'))['total'] or 0
            chart_data.append(float(total))
            chart_labels.append(f"{LITHUANIAN_MONTHS[current.month]} {current.year}")

            current = month_end + timedelta(days=1)

    metrics = payments.aggregate(
        total_revenue=Sum('amount', filter=Q(status='paid')),
        successful_transactions=Count('id', filter=Q(status='paid')),
        pending_invoices=Count('id', filter=Q(status='pending')),
        overdue_invoices=Count('id', filter=Q(status='overdue')),
        outstanding=Sum('amount', filter=Q(status='pending')),
        overdue_amount=Sum('amount', filter=Q(status='overdue')),
        canceled_payments=Count('id', filter=Q(status='cancelled')),
        canceled_amount=Sum('amount', filter=Q(status='cancelled')),
    )

    total_revenue = metrics['total_revenue'] or 0
    successful_transactions = metrics['successful_transactions'] or 0
    pending_invoices = metrics['pending_invoices'] or 0
    overdue_invoices = metrics['overdue_invoices'] or 0
    outstanding = metrics['outstanding'] or 0
    overdue_amount = metrics['overdue_amount'] or 0
    canceled_payments = metrics['canceled_payments'] or 0
    canceled_amount = metrics['canceled_amount'] or 0
    total_payments = payments.count()

    revenue_change_percent = 0
    if chart_data:
        if len(chart_data) > 1:
            last = chart_data[-1]
            prev = chart_data[-2]
            if prev > 0:
                revenue_change_percent = ((last - prev) / prev) * 100

    financial_score = 80
    if total_revenue > 10000:
        financial_score += 10
    elif total_revenue > 5000:
        financial_score += 5

    if successful_transactions > 20:
        financial_score += 10
    elif successful_transactions > 10:
        financial_score += 5

    if pending_invoices > 5:
        financial_score -= 10
    elif pending_invoices > 3:
        financial_score -= 5

    if overdue_invoices > 0:
        financial_score -= 15

    financial_score = max(0, min(100, financial_score))

    formatted_date_start = format_date_lt(date_start) if date_start else ""
    formatted_date_end = format_date_lt(date_end) if date_end else ""
    all_tags = Tag.objects.filter(user=request.user)

    expense_date_start_str = request.GET.get('expense_date_start')
    expense_date_end_str = request.GET.get('expense_date_end')

    expenses = Expense.objects.filter(user=request.user).order_by('-date')

    expense_date_start = None
    expense_date_end = None
    expense_all_time = False

    if expense_date_start_str:
        try:
            expense_date_start = datetime.datetime.strptime(expense_date_start_str, '%Y-%m-%d').date()
        except:
            pass

    if expense_date_end_str:
        try:
            expense_date_end = datetime.datetime.strptime(expense_date_end_str, '%Y-%m-%d').date()
        except:
            pass

    if not expense_date_start_str and not expense_date_end_str:
        expense_all_time = True
        if expenses.exists():
            expense_date_start = expenses.aggregate(min_date=Min('date'))['min_date']
            expense_date_end = expenses.aggregate(max_date=Max('date'))['max_date']
        else:
            today = timezone.now().date()
            expense_date_start = today - timedelta(days=29)
            expense_date_end = today
    else:
        if expense_date_start and expense_date_end:
            expenses = expenses.filter(date__range=[expense_date_start, expense_date_end])
        elif expense_date_start:
            expenses = expenses.filter(date__gte=expense_date_start)
        elif expense_date_end:
            expenses = expenses.filter(date__lte=expense_date_end)

    if not expense_date_start or not expense_date_end:
        if expenses.exists():
            expense_date_start = expenses.aggregate(min_date=Min('date'))['min_date']
            expense_date_end = expenses.aggregate(max_date=Max('date'))['max_date']
        else:
            today = timezone.now().date()
            expense_date_start = today - timedelta(days=29)
            expense_date_end = today

    expense_days_diff = (expense_date_end - expense_date_start).days

    expense_chart_data = []
    expense_chart_labels = []

    if expense_days_diff <= 30:
        for i in range(expense_days_diff + 1):
            current_date = expense_date_start + timedelta(days=i)
            day_expenses = expenses.filter(date=current_date)
            total = day_expenses.aggregate(total=Sum('amount'))['total'] or 0
            expense_chart_data.append(float(total))
            expense_chart_labels.append(f"{current_date.day} {LITHUANIAN_MONTHS[current_date.month]}")
    elif expense_days_diff <= 90:
        current = expense_date_start
        while current <= expense_date_end:
            week_end = current + timedelta(days=6)
            if week_end > expense_date_end:
                week_end = expense_date_end

            week_expenses = expenses.filter(date__gte=current, date__lte=week_end)
            total = week_expenses.aggregate(total=Sum('amount'))['total'] or 0
            expense_chart_data.append(float(total))
            start_str = f"{current.day} {LITHUANIAN_MONTHS[current.month]}"
            end_str = f"{week_end.day} {LITHUANIAN_MONTHS[week_end.month]}"
            expense_chart_labels.append(f"{start_str} - {end_str}")

            current = week_end + timedelta(days=1)
    else:
        current = expense_date_start.replace(day=1)
        while current <= expense_date_end:
            next_month = current.replace(day=28) + timedelta(days=4)
            month_end = next_month - timedelta(days=next_month.day)
            if month_end > expense_date_end:
                month_end = expense_date_end

            month_expenses = expenses.filter(date__gte=current, date__lte=month_end)
            total = month_expenses.aggregate(total=Sum('amount'))['total'] or 0
            expense_chart_data.append(float(total))
            expense_chart_labels.append(f"{LITHUANIAN_MONTHS[current.month]} {current.year}")

            current = month_end + timedelta(days=1)

    search = request.GET.get('search')
    if search:
        expenses = expenses.filter(description__icontains=search)

    tags_filter = request.GET.getlist('tags')
    selected_tag_ids = [int(tag_id) for tag_id in tags_filter if tag_id.isdigit()]
    if selected_tag_ids:
        expenses = expenses.filter(tags__id__in=selected_tag_ids).distinct()

    tag_summary = []
    total_expenses = 0
    expenses_count = expenses.count()

    untagged_expenses = expenses.filter(tags__isnull=True)
    untagged_list = []
    for exp in untagged_expenses:
        exp.formatted_date = format_date_lt(exp.date)
        untagged_list.append(exp)

    untagged_total = untagged_expenses.aggregate(total=Sum('amount'))['total'] or 0
    if untagged_total > 0:
        tag_summary.append({
            'id': None,
            'name': 'Be žymos',
            'color': '#6c757d',
            'total': untagged_total,
            'expenses': untagged_list
        })
        total_expenses += untagged_total

    for tag in all_tags:
        tagged_expenses = expenses.filter(tags=tag)
        tagged_list = []
        for exp in tagged_expenses:
            exp.formatted_date = format_date_lt(exp.date)
            tagged_list.append(exp)

        tag_total = tagged_expenses.aggregate(total=Sum('amount'))['total'] or 0
        if tag_total > 0:
            tag_summary.append({
                'id': tag.id,
                'name': tag.name,
                'color': tag.color,
                'total': tag_total,
                'expenses': tagged_list
            })
            total_expenses += tag_total

    tag_summary.sort(key=lambda x: x['total'], reverse=True)

    expense_formatted_date_start = format_date_lt(expense_date_start) if expense_date_start else ""
    expense_formatted_date_end = format_date_lt(expense_date_end) if expense_date_end else ""

    context = {
        'chart_labels': json.dumps(chart_labels),
        'chart_data': json.dumps(chart_data),
        'revenue_change_percent': revenue_change_percent,
        'total_revenue': total_revenue,
        'successful_transactions': successful_transactions,
        'pending_invoices': pending_invoices,
        'overdue_invoices': overdue_invoices,
        'outstanding': outstanding,
        'overdue_amount': overdue_amount,
        'canceled_payments': canceled_payments,
        'canceled_amount': canceled_amount,
        'financial_score': financial_score,
        'selected_statuses': selected_statuses,
        'date_start': date_start.strftime('%Y-%m-%d') if date_start else "",
        'date_end': date_end.strftime('%Y-%m-%d') if date_end else "",
        'formatted_date_start': formatted_date_start,
        'formatted_date_end': formatted_date_end,
        'total_payments': total_payments,
        'all_time': all_time,
        'days_diff': days_diff,
        'tags': all_tags,
        'tag_summary': tag_summary,
        'total_expenses': total_expenses,
        'expenses_count': expenses_count,
        'selected_tags': selected_tag_ids,
        'expense_date_start': expense_date_start.strftime('%Y-%m-%d') if expense_date_start else "",
        'expense_date_end': expense_date_end.strftime('%Y-%m-%d') if expense_date_end else "",
        'expense_all_time': expense_all_time,
        'expense_formatted_date_start': expense_formatted_date_start,
        'expense_formatted_date_end': expense_formatted_date_end,
        'expense_chart_labels': json.dumps(expense_chart_labels),
        'expense_chart_data': json.dumps(expense_chart_data),
        'expense_days_diff': expense_days_diff,
    }

    return render(request, 'dashboard.html', context)


@login_required
def profile_view(request):
    user = request.user
    if request.method == 'POST':
        if 'delete_logo' in request.POST:
            if user.company_logo:
                user.company_logo.delete(save=False)
                user.company_logo = None
        user.first_name = request.POST.get('first_name', '')
        user.last_name = request.POST.get('last_name', '')
        user.email = request.POST.get('email', '')
        user.phone = request.POST.get('phone', '')
        user.website = request.POST.get('website', '')
        user.company = request.POST.get('company', '')
        user.position = request.POST.get('position', '')
        user.address = request.POST.get('address', '')
        user.bio = request.POST.get('bio', '')
        user.save()
        messages.success(request, 'Profile updated successfully!')
        return redirect('profile')

    return render(request, 'profile.html', {'user': user})


from django.shortcuts import render


@login_required
def payment_list(request):
    payments = Payment.objects.filter(user=request.user).order_by('-date_issued')

    filter_params = request.GET.copy()
    if 'page' in filter_params:
        del filter_params['page']

    selected_statuses = request.GET.getlist('status', [])
    search_query = request.GET.get('search', '')
    client_id = request.GET.get('client', '')
    date_start = request.GET.get('date_start', '')
    date_end = request.GET.get('date_end', '')
    min_amount = request.GET.get('min_amount', '')
    max_amount = request.GET.get('max_amount', '')

    if selected_statuses:
        payments = payments.filter(status__in=selected_statuses)

    if search_query:
        payments = payments.filter(
            Q(client__name__icontains=search_query) |
            Q(invoice_number__icontains=search_query)
        )

    if client_id:
        payments = payments.filter(client_id=client_id)

    if date_start:
        payments = payments.filter(date_issued__gte=date_start)
    if date_end:
        payments = payments.filter(date_issued__lte=date_end)

    if min_amount:
        payments = payments.filter(amount__gte=min_amount)
    if max_amount:
        payments = payments.filter(amount__lte=max_amount)

    paginator = Paginator(payments, 10)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    clients = Partner.objects.filter(user=request.user)

    context = {
        'payments': page_obj,
        'clients': clients,
        'selected_statuses': selected_statuses,
        'filter_params': filter_params.urlencode(),
        'request': request,
    }
    return render(request, 'payments.html', context)


@login_required
def payments(request):
    payments = Payment.objects.filter(user=request.user).order_by('-date_issued')
    clients = Partner.objects.filter(user=request.user)

    paginator = Paginator(payments, 10)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    return render(request, 'payments.html', {
        'payments': page_obj,
        'clients': clients,
        'day_today': datetime.date.today(),
        'company_name': request.user.company or request.user.get_full_name()  # Добавляем
    })


@login_required
def register_support_fonts(request):
    try:
        font_locations = [
            # Windows
            '/home/apskaita/public_html/finance/back/fonts/DejaVuSans.ttf',  # Arial Unicode MS
            'C:/Windows/Fonts/times.ttf',  # Arial Unicode MS',  # Times New Roman

            # Linux
            '/usr/share/fonts/truetype/dejavu/',
            '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',

            # MacOS
            '/Library/Fonts/Arial Unicode.ttf',

            # В проекте
            os.path.join(os.path.dirname(__file__), 'fonts\\arialuni.ttf'),
            os.path.join(os.path.dirname(__file__), 'fonts\\times.ttf'),
        ]
        print('new')
        for font_path in font_locations:
            print(2)
            if os.path.exists(font_path):
                print(font_path)
                font_name = os.path.splitext(os.path.basename(font_path))[0]
                try:
                    pdfmetrics.registerFont(TTFont(font_name, font_path))
                    bold_path = '/home/apskaita/public_html/finance/back/fonts/arial_bolditalicmt.ttf'
                    if os.path.exists(bold_path):
                        pdfmetrics.registerFont(TTFont(f'{font_name}-Bold', bold_path))
                    return font_name
                except:
                    continue

        return 'Helvetica'
    except Exception as e:
        return 'Helvetica'


@login_required
def generate_invoice_pdf(request, payment_id):
    payment = get_object_or_404(Payment, id=payment_id, user=request.user)

    custom_invoice_number = request.GET.get('invoice_number')
    if custom_invoice_number:
        payment.invoice_number = custom_invoice_number

    client = payment.client

    def get_lithuanian_month(month):
        months = {
            1: "sausio", 2: "vasario", 3: "kovo", 4: "balandžio",
            5: "gegužės", 6: "birželio", 7: "liepos", 8: "rugpjūčio",
            9: "rugsėjo", 10: "spalio", 11: "lapkričio", 12: "gruodžio"
        }
        return months.get(month, "")

    support_font = register_support_fonts(request)

    buffer = BytesIO()

    doc = SimpleDocTemplate(
        buffer,
        pagesize=A4,
        leftMargin=20 * mm,
        rightMargin=20 * mm,
        topMargin=10 * mm,
        bottomMargin=10 * mm,
        title=f"Sąskaita {payment.invoice_number}"
    )

    styles = getSampleStyleSheet()

    style_heading = ParagraphStyle(
        'Heading1',
        parent=styles['Heading1'],
        alignment=TA_LEFT,
        fontSize=14,
        spaceAfter=5,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold'
    )

    style_normal = ParagraphStyle(
        'Normal',
        parent=styles['BodyText'],
        fontSize=10,
        spaceAfter=3,
        fontName=support_font
    )

    style_bold = ParagraphStyle(
        'Bold',
        parent=style_normal,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold'
    )

    style_table_header = ParagraphStyle(
        'TableHeader',
        parent=style_normal,
        fontName=f'{support_font}-Bold' if support_font != 'Helvetica' else 'Helvetica-Bold',
        fontSize=9,
        alignment=TA_CENTER
    )

    style_table_cell = ParagraphStyle(
        'TableCell',
        parent=style_normal,
        fontSize=9
    )

    style_table_cell_center = ParagraphStyle(
        'TableCellCenter',
        parent=style_table_cell,
        alignment=TA_CENTER
    )

    style_table_cell_right = ParagraphStyle(
        'TableCellRight',
        parent=style_table_cell,
        alignment=TA_RIGHT
    )

    def format_currency(value):
        try:
            value = float(value)
            return f"{value:,.2f}".replace(",", " ").replace(".", ",")
        except:
            return "0,00"

    story = []

    logo = None
    if request.user.company_logo:
        try:
            with default_storage.open(request.user.company_logo.name, 'rb') as logo_file:
                logo_data = BytesIO(logo_file.read())
                ext = request.user.company_logo.name.split('.')[-1].lower()
                if ext in ['jpg', 'jpeg', 'png']:
                    logo = Image(logo_data, width=100, height=30)
        except Exception as e:
            print(f"Logo error: {e}")

    header_table_data = []
    template = payment.template
    name = template.name

    if logo:
        header_table_data.append([Paragraph(name, style_heading), logo])
    else:
        header_table_data.append([Paragraph(name, style_heading), ""])

    header_table = Table(header_table_data, colWidths=[120 * mm, 50 * mm])
    header_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('ALIGN', (0, 0), (0, 0), 'LEFT'),
        ('ALIGN', (1, 0), (1, 0), 'RIGHT'),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 10),
    ]))
    story.append(header_table)
    story.append(Spacer(1, 5))

    issued_month_lt = get_lithuanian_month(payment.date_issued.month)
    number_date_data = [
        [
            Paragraph(f"<b>Serija ir Nr.:</b> {payment.invoice_number}", style_normal),
            Paragraph(
                f"<b>Išrašymo data:</b> {payment.date_issued.year} m. {issued_month_lt} {payment.date_issued.day} d.",
                style_normal
            )
        ]
    ]
    number_date_table = Table(number_date_data, colWidths=[90 * mm, 80 * mm])
    number_date_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('ALIGN', (0, 0), (0, 0), 'LEFT'),
        ('ALIGN', (1, 0), (1, 0), 'RIGHT'),
    ]))
    story.append(number_date_table)
    story.append(Spacer(1, 15))

    seller_info = [Paragraph("<b>Pardavėjas</b>", style_bold),
                   Paragraph(request.user.company or f"{request.user.first_name} {request.user.last_name}",
                             style_normal)]
    if request.user.address:
        seller_info.append(Paragraph(f"Adresas: {request.user.address}", style_normal))
    if request.user.registration_number:
        seller_info.append(Paragraph(f"Įmonės kodas: {request.user.registration_number}", style_normal))
    if request.user.vat_code:
        seller_info.append(Paragraph(f"PVM kodas: {request.user.vat_code}", style_normal))
    if request.user.website:
        seller_info.append(Paragraph(f"Svetainė: {request.user.website}", style_normal))
    if request.user.email_company:
        seller_info.append(Paragraph(f"El. paštas: {request.user.email_company}", style_normal))
    if request.user.address:
        seller_info.append(Paragraph(f"Telefonas: {request.user.phone}", style_normal))
    if request.user.bank_name:
        seller_info.append(Paragraph(f"Banko pavadinimas: {request.user.bank_name}", style_normal))
    if request.user.bank_account:
        seller_info.append(Paragraph(f"Banko sąskaita: {request.user.bank_account}", style_normal))
    if request.user.bic:
        seller_info.append(Paragraph(f"BIC/SWIFT kodas: {request.user.bic}", style_normal))

    buyer_info = [Paragraph("<b>Pirkėjas</b>", style_bold)]
    if client:
        buyer_info.append(Paragraph(client.name, style_normal))
        if client.address:
            buyer_info.append(Paragraph(f"Adresas: {client.address}", style_normal))
        if client.registration_number:
            buyer_info.append(Paragraph(f"Įmonės kodas: {client.registration_number}", style_normal))
        if client.vat_code:
            buyer_info.append(Paragraph(f"PVM kodas: {client.vat_code}", style_normal))
        if client.email:
            buyer_info.append(Paragraph(f"El. paštas: {client.email}", style_normal))
        if client.contact_person:
            buyer_info.append(Paragraph(f"Kontaktinis asmuo: {client.contact_person}", style_normal))
        if client.phone:
            buyer_info.append(Paragraph(f"Telefonas: {client.phone}", style_normal))
    else:
        buyer_info.append(Paragraph("Nenurodytas", style_normal))

    # Create bordered boxes using table
    seller_buyer_data = [[seller_info, buyer_info]]
    seller_buyer_table = Table(
        seller_buyer_data,
        colWidths=[85 * mm, 85 * mm]
    )

    # Box styling with borders
    seller_buyer_table.setStyle(TableStyle([
        ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ('BOX', (0, 0), (0, 0), 1, colors.black),  # Left box border
        ('BOX', (1, 0), (1, 0), 1, colors.black),  # Right box border
        ('PADDING', (0, 0), (-1, -1), 5),
        ('LEFTPADDING', (0, 0), (-1, -1), 3),
        ('RIGHTPADDING', (0, 0), (-1, -1), 3),
        ('TOPPADDING', (0, 0), (-1, -1), 5),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 5),
    ]))

    story.append(seller_buyer_table)
    story.append(Spacer(1, 15))

    # 4. Items table
    table_data = [
        [
            Paragraph("Pavadinimas", style_table_header),
            Paragraph("Vnt. kaina (be PVM)", style_table_header),
            Paragraph("Kiekis", style_table_header),
            Paragraph("Matas", style_table_header),
            Paragraph("Suma be PVM", style_table_header),
        ]
    ]

    # Add discount column if needed
    no_discount = True
    if any([i.get('discount', 0) for i in payment.items]):
        no_discount = False
        table_data[0].insert(4, Paragraph("Nuolaida", style_table_header))

    no_vat = True
    if any([i.get('vat', 0) for i in payment.items]):
        no_vat = False
        table_data[0].insert(1, Paragraph("PVM", style_table_header))

    # Calculate column widths
    col_widths = [70 * mm, 25 * mm, 15 * mm, 20 * mm, 25 * mm]
    if not no_discount:
        col_widths.insert(4, 15 * mm)  # Discount column
    if not no_vat:
        col_widths.insert(1, 15 * mm)  # Discount column
    # Add items
    vpm = 0
    pre_total_to_tittle = 0
    for item in payment.items:
        name = item.get('name', '')
        quantity = item.get('quantity', 1)
        unit = item.get('unit', '')
        vat = item.get('vat', '')
        disc = item.get('discount', 0)
        price = item.get('unit_price', 0)
        pre_total = price * quantity
        pre_total2 = pre_total - (pre_total / 100 * disc)
        pre_total_to_tittle += pre_total2
        vpm += (pre_total2 * (item.get('vat', 0) / 100))

        row = [
            Paragraph(name, style_table_cell_center),
            Paragraph(f"{format_currency(price)} €", style_table_cell_center),
            Paragraph(str(quantity), style_table_cell_center),
            Paragraph(unit, style_table_cell_center),
            Paragraph(f"{format_currency(pre_total2)} €", style_table_cell_center),
        ]

        if not no_discount:
            row.insert(4, Paragraph(f"{disc}%", style_table_cell_center))
        if not no_vat:
            row.insert(1, Paragraph(f"{vat}%", style_table_cell_center))
        table_data.append(row)

    # Create items table
    items_table = Table(table_data, colWidths=col_widths)
    items_table.setStyle(TableStyle([
        ('GRID', (0, 0), (-1, -1), 1, colors.black),
        ('FONTSIZE', (0, 0), (-1, -1), 9),
        ('BOTTOMPADDING', (0, 0), (-1, 0), 5),
        ('TOPPADDING', (0, 0), (-1, 0), 5),
        ('ALIGN', (2, 0), (-1, -1), 'RIGHT'),
        ('ALIGN', (1, 0), (1, -1), 'CENTER'),
        ('ALIGN', (3, 0), (4, -1), 'CENTER'),
    ]))
    story.append(items_table)
    story.append(Spacer(1, 10))

    # 5. Totals
    total_amount = float(payment.amount)
    if not no_vat:
        story.append(Paragraph(f"Iš viso (be PVM): {format_currency(pre_total_to_tittle)} €", style_table_cell_right))
        story.append(Paragraph(f"PVM: {format_currency(vpm)} €", style_table_cell_right))
        story.append(
            Paragraph(f"Iš viso (su PVM): {format_currency(pre_total_to_tittle + vpm)} €", style_table_cell_right))
    else:
        story.append(Paragraph(f"Iš viso: {format_currency(pre_total_to_tittle + vpm)} €", style_table_cell_right))
    story.append(Spacer(1, 10))
    if payment.description:
        # . Description and payment terms
        story.append(Paragraph(f"Aprašymas: {payment.description}", style_normal))

    # 7. Amount in words
    try:
        amount_in_words = num2words(
            total_amount,
            lang='lt',
            to='currency',
            currency='EUR'
        ).capitalize().replace('eurai', 'Eur').replace('euro', 'Eur')
        story.append(Paragraph(f"Suma žodžiais: {amount_in_words}", style_normal))
    except Exception as e:
        print(f"Num2words error: {e}")

    if payment.date_due:
        due_month_lt = get_lithuanian_month(payment.date_due.month)
        story.append(Paragraph(
            f"Apmokėti iki: {payment.date_due.year} m. {due_month_lt} {payment.date_due.day} d.",
            style_normal
        ))
    story.append(Paragraph(f"Sąskaitą išrašė: {request.user.first_name} {request.user.last_name}", style_normal))

    story.append(Spacer(1, 15))

    # 8. Service image on new page
    service_image = None
    if payment.image:
        try:
            with default_storage.open(payment.image.name, 'rb') as img_file:
                img_data = BytesIO(img_file.read())

                # Resize with PIL
                try:
                    from PIL import Image as PILImage
                    pil_img = PILImage.open(img_data)
                    img_width, img_height = pil_img.size

                    # Calculate proportional size
                    max_width = 180 * mm
                    max_height = 200 * mm
                    width_ratio = max_width / img_width
                    height_ratio = max_height / img_height
                    scale_ratio = min(width_ratio, height_ratio, 1.0)

                    new_width = img_width * scale_ratio
                    new_height = img_height * scale_ratio

                    img_data.seek(0)
                    service_image = Image(img_data, width=new_width, height=new_height)
                except:
                    img_data.seek(0)
                    service_image = Image(img_data, width=150 * mm, height=100 * mm)
        except Exception as e:
            print(f"Service image error: {e}")

    if service_image:
        story.append(PageBreak())  # Force new page
        story.append(Spacer(1, 5))
        service_image.hAlign = 'CENTER'
        story.append(service_image)

    # Generate PDF
    doc.build(story)

    # Prepare response
    company_name = payment.client.name
    safe_company_name = re.sub(r'[^\w\sąčęėįšųūžĄČĘĖĮŠŲŪŽ]', '', company_name)
    safe_company_name = re.sub(r'\s+', '_', safe_company_name)[:50]
    filename = f"{safe_company_name} Faktūra.pdf"

    buffer.seek(0)
    response = HttpResponse(buffer, content_type='application/pdf')

    print(request.GET)
    # Только для предпросмотра
    if 'preview' in request.GET:
        response['Content-Disposition'] = f'inline; filename="{filename}"'
    # Для скачивания
    elif 'download' in request.GET:
        response['Content-Disposition'] = f'attachment; filename="{filename}"'
    # По умолчанию (должно использоваться только для скачивания через JS)


    return response


def profile(request):
    return render(request, template_name='profile.html')


@login_required
@require_POST
def delete_company_logo(request):
    user = request.user
    if user.company_logo:
        # Удаляем файл изображения
        user.company_logo.delete()
        # Устанавливаем поле company_logo в None
        user.company_logo = None
        user.save()
        return JsonResponse({'status': 'success'})
    return JsonResponse({'status': 'error', 'message': 'Logo not found'}, status=400)


# views.py
from django.views.generic import TemplateView
from django.core.paginator import Paginator
from django.views.generic import TemplateView
from django.core.paginator import Paginator