# views.py
from django.db.models import Q
from django.shortcuts import render
from rest_framework.generics import get_object_or_404
from rest_framework import generics, pagination, status
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated  # Добавлен импорт
from partners.models import Partner
from partners.serializers import SerializerPartners
from django.contrib.auth.decorators import login_required
from rest_framework.parsers import MultiPartParser, FormParser
from .models import Document
from .serializers import DocumentSerializer

class DocumentListView(generics.ListAPIView):
    permission_classes = [IsAuthenticated]
    pagination_class = None
    serializer_class = DocumentSerializer

    def get_queryset(self):
        partner_id = self.kwargs['partner_id']
        partner = get_object_or_404(Partner, id=partner_id, user=self.request.user)
        return Document.objects.filter(partner=partner)


class DocumentUploadView(generics.CreateAPIView):
    permission_classes = [IsAuthenticated]
    queryset = Document.objects.all()
    serializer_class = DocumentSerializer
    parser_classes = [MultiPartParser, FormParser]

    def perform_create(self, serializer):
        partner_id = self.kwargs['partner_id']
        partner = get_object_or_404(Partner, id=partner_id, user=self.request.user)
        serializer.save(partner=partner)


class DocumentDeleteView(generics.DestroyAPIView):
    permission_classes = [IsAuthenticated]
    queryset = Document.objects.all()

    def get_object(self):
        doc_id = self.kwargs['doc_id']
        return get_object_or_404(
            Document, 
            id=doc_id, 
            partner__user=self.request.user
        )


class StandardPagination(pagination.PageNumberPagination):
    page_size = 10
    page_size_query_param = 'page_size'
    max_page_size = 20

    def get_paginated_response(self, data):
        return Response({
            'count': self.page.paginator.count,
            'results': data,
            'total_pages': self.page.paginator.num_pages,
            'current_page': self.page.number,
        })


class ModelViewListCreatePartners(generics.ListCreateAPIView):
    permission_classes = [IsAuthenticated]
    serializer_class = SerializerPartners
    queryset = Partner.objects.all()
    pagination_class = StandardPagination

    def get_queryset(self):
        if not self.request.user.is_authenticated:
            return Partner.objects.none()
            
        queryset = Partner.objects.filter(user=self.request.user).order_by('id')

        filter_fields = {
            'name': 'name__icontains',
            'contact_person': 'contact_person__icontains',
            'email': 'email__icontains',
            'phone': 'phone__icontains',
            'address': 'address__icontains',
            'vat_code': 'vat_code__icontains',
            'registration_number': 'registration_number__icontains'
        }

        # Создаем Q-объект для фильтрации
        q_objects = Q()

        for param, field_lookup in filter_fields.items():
            value = self.request.query_params.get(param)
            if value:
                q_objects &= Q(**{field_lookup: value})

        return queryset.filter(q_objects)

    def create(self, request, *args, **kwargs):
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
            
        data = request.data.copy()
        data['user'] = request.user.id
        serializer = self.get_serializer(data=data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)
        headers = self.get_success_headers(serializer.data)
        return Response(serializer.data, status=status.HTTP_201_CREATED, headers=headers)


class ModelViewRetrieveUpdateDestroyPartners(generics.RetrieveUpdateDestroyAPIView):
    permission_classes = [IsAuthenticated] 
    serializer_class = SerializerPartners
    queryset = Partner.objects.all()

    def get_queryset(self):
        if not self.request.user.is_authenticated:
            return Partner.objects.none()
        return self.queryset.filter(user=self.request.user)

    def update(self, request, *args, **kwargs):
        if not request.user.is_authenticated:
            return Response(
                {"detail": "Требуется авторизация"},
                status=status.HTTP_401_UNAUTHORIZED
            )
            
        pk = kwargs.get('pk', None)
        if not pk:
            return Response(status=status.HTTP_400_BAD_REQUEST)

        data = request.data
        data['user'] = request.user.id
        instance = get_object_or_404(self.get_queryset(), pk=pk)
        serializer = self.get_serializer(instance, data=data)
        serializer.is_valid(raise_exception=True)
        self.perform_update(serializer)

        return Response(serializer.data)

@login_required
def partners(request):
    return render(request, template_name='partners.html')